import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * 企信通短信发送客户端（HTTP POST UTF-8接口）
 * 对接文档：短信接口Http1.0_明文密码.doc
 * 功能：发送普通短信（含签名），支持多号码
 */
public class SmsSender {

    // =============== 配置参数（请根据实际情况修改）===============
    private static final String SEND_URL = "http://cloud.thinksms.com/sms.aspx"; // UTF-8 接口地址
    private static final String USER_ID = "your_user_id";        // （登录后台->客户管理->我的信息）查询企业ID
    private static final String ACCOUNT = "your_account";        // 短信账号
    private static final String PASSWORD = "your_password";      // 登录密码
    // ============================================================

    /**
     * 发送短信主方法
     *
     * @param mobile  接收号码，多个用英文逗号分隔，如 "13800138000,13900139000"
     * @param content 短信内容（必须包含签名，如：【智想云创】您的注册验证码：8899）
     * @return 接口原始返回的 JSON 字符串
     */
    public static String sendSms(String mobile, String content) {
        HttpURLConnection connection = null;
        OutputStream outputStream = null;
        InputStream inputStream = null;
        BufferedReader reader = null;

        try {
            // 创建连接
            URL url = new URL(SEND_URL);
            connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setDoInput(true);
            connection.setUseCaches(false);
            connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");

            // 构建请求参数（使用 UTF-8 编码）
            StringBuilder params = new StringBuilder();
            params.append("action=send")
                  .append("&rt=json") // 返回格式为 JSON
                  .append("&userid=").append(URLEncoder.encode(USER_ID, StandardCharsets.UTF_8))
                  .append("&account=").append(URLEncoder.encode(ACCOUNT, StandardCharsets.UTF_8))
                  .append("&password=").append(URLEncoder.encode(PASSWORD, StandardCharsets.UTF_8))
                  .append("&mobile=").append(URLEncoder.encode(mobile, StandardCharsets.UTF_8))
                  .append("&content=").append(URLEncoder.encode(content, StandardCharsets.UTF_8));

            // 发送请求
            byte[] postData = params.toString().getBytes(StandardCharsets.UTF_8);
            connection.getOutputStream().write(postData);
            connection.getOutputStream().flush();

            // 读取响应
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpURLConnection.HTTP_OK) {
                inputStream = connection.getInputStream();
            } else {
                inputStream = connection.getErrorStream(); // 尝试读取错误流
            }

            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }

            // 返回原始 JSON 结果
            return response.toString();

        } catch (Exception e) {
            // 默认不处理异常，返回错误信息（实际项目中建议记录日志）
            return "{\"ReturnStatus\":\"Faild\",\"Message\":\"" + e.getMessage() + "\"}";
        } finally {
            // 关闭资源
            try { if (reader != null) reader.close(); } catch (IOException ignored) {}
            try { if (inputStream != null) inputStream.close(); } catch (IOException ignored) {}
            try { if (outputStream != null) outputStream.close(); } catch (IOException ignored) {}
            if (connection != null) connection.disconnect();
        }
    }

    // ==================== 示例调用 ====================
    public static void main(String[] args) {
        // 示例：发送验证码短信
        String mobile = "13800138000"; // 替换为真实测试号码
        String content = "【智想云创】您的注册验证码：8899"; // 必须包含签名

        String result = sendSms(mobile, content);
        System.out.println("接口返回结果: " + result);

        /*
         * ==================== 返回值解析（供开发者参考）====================
         *
         * 成功返回示例：
         * {"ReturnStatus":"Success","Message":"ok","RemainPoint":109346,"TaskID":365,"SuccessCounts":1}
         *
         * 字段说明：
         * - ReturnStatus: 返回状态，"Success" 表示请求提交成功（不代表短信送达）
         * - Message: 提示信息，"ok" 表示成功，其他为错误原因
         * - RemainPoint: 当前账户剩余短信条数
         * - TaskID: 本次发送任务的唯一 ID，可用于查询状态报告
         * - SuccessCounts: 成功提交的手机号数量（可用于校验）
         *
         * ===================================================================
         */
    }
}