import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * 企信通短信发送客户端（GBK编码版本）
 * 使用场景：对接系统默认编码为GBK（如部分Windows服务器）
 * 接口文档：短信接口Http1.0_明文密码.doc
 */
public class SmsSenderGBK {

    // =============== 配置参数（请根据实际情况修改）===============
    private static final String SEND_URL = "http://cloud.thinksms.com/smsGBK.aspx"; // GBK接口地址
    private static final String USER_ID = "your_user_id";        // （登录后台->客户管理->我的信息）查询企业ID
    private static final String ACCOUNT = "your_account";        // 短信账号
    private static final String PASSWORD = "your_password";      // 登录密码
    // ============================================================

    /**
     * 发送短信（GBK编码）
     *
     * @param mobile  接收号码，多个用英文逗号分隔："13800138000,13900139000"
     * @param content 短信内容（必须含签名，如：【智想云创】您的注册验证码：8899）
     * @return 接口原始返回的 JSON 字符串（UTF-8 编码！注意：返回值是UTF-8）
     */
    public static String sendSms(String mobile, String content) {
        HttpURLConnection connection = null;
        OutputStream outputStream = null;
        InputStream inputStream = null;
        BufferedReader reader = null;

        try {
            URL url = new URL(SEND_URL);
            connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setDoInput(true);
            connection.setUseCaches(false);
            connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded;charset=GBK");

            // 构建请求参数（关键：使用 GBK 编码）
            StringBuilder params = new StringBuilder();
            params.append("action=send")
                  .append("&rt=json") // 返回格式为 JSON（返回内容为 UTF-8）
                  .append("&userid=").append(URLEncoder.encode(USER_ID, "GBK"))
                  .append("&account=").append(URLEncoder.encode(ACCOUNT, "GBK"))
                  .append("&password=").append(URLEncoder.encode(PASSWORD, "GBK"))
                  .append("&mobile=").append(URLEncoder.encode(mobile, "GBK"))
                  .append("&content=").append(URLEncoder.encode(content, "GBK"));

            // 发送请求体（GBK编码）
            byte[] postData = params.toString().getBytes("GBK");
            outputStream = connection.getOutputStream();
            outputStream.write(postData);
            outputStream.flush();

            // 读取响应（注意：返回的JSON是UTF-8编码，不是GBK）
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpURLConnection.HTTP_OK) {
                inputStream = connection.getInputStream();
            } else {
                inputStream = connection.getErrorStream(); // 获取错误信息
            }

            // 无论接口编码如何，返回的JSON建议用UTF-8解析（文档中返回值为标准JSON）
            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }

            return response.toString();

        } catch (Exception e) {
            // 默认返回错误信息（实际项目建议记录日志）
            return "{\"ReturnStatus\":\"Faild\",\"Message\":\"" + e.getMessage() + "\"}";
        } finally {
            // 关闭资源
            try { if (reader != null) reader.close(); } catch (IOException ignored) {}
            try { if (inputStream != null) inputStream.close(); } catch (IOException ignored) {}
            try { if (outputStream != null) outputStream.close(); } catch (IOException ignored) {}
            if (connection != null) connection.disconnect();
        }
    }

    // ==================== 示例调用 ====================
    public static void main(String[] args) {
        String mobile = "13800138000"; // 替换为测试号码
        String content = "【智想云创】您好，您的订单已发货，请注意查收！";

        String result = sendSms(mobile, content);
        System.out.println("接口返回结果: " + result);

        /*
         * ==================== 返回值解析（开发者参考）====================
         *
         * 成功示例：
         * {"ReturnStatus":"Success","Message":"ok","RemainPoint":98765,"TaskID":402,"SuccessCounts":1}
         *
         * 字段说明：
         * - ReturnStatus: "Success" 表示请求提交成功
         * - Message: "ok" 表示正常，其他为错误提示
         * - RemainPoint: 剩余短信条数
         * - TaskID: 本次任务ID，用于查询状态报告
         * - SuccessCounts: 成功提交的手机号数量
         *
         * ===================================================================
         */
    }
}